#!/bin/bash

[ -f /etc/os-release ] && . /etc/os-release


INST_DIR="/opt/assistant"
SHELLS=$INST_DIR/bin/shells.ini
LOG_FILE=$INST_DIR/log/setup.log
SHORTCUT_PATH=$INST_DIR/scripts/assistant.desktop
USERS_LIST=$(ls -d /home/*/)
PATH_TO_CONF=/.config/user-dirs.dirs
TRUST="/etc/X11/trusted"
TRUST_D="/etc/X11/trusted.d"
CERT_PATH=/etc/assistant/cert/ca-cert-native
PATHS_TO_CERT=(
	"/etc/ssl/certs/ca-certificates.crt" 
	"/etc/pki/tls/certs/ca-bundle.crt" 
	"/etc/pki/ca-trust/extracted/pem/tls-ca-bundle.pem" 
	"/usr/share/ca-certificates/ca-bundle.crt" 
	"/etc/ssl/certs/ssl-cert-snakeoil.pem" 
	"/etc/ca-certificates/extracted/ca-bundle.trust.crt"
	"/var/lib/ca-certificates/ca-bundle.pem"
	)
PAM_LIGHTDM_SETTINGS=(
	"/etc/pam.d/lightdm-autologin"
	"/etc/pam.d/lightdm"
	)


log(){
	if [ -n "$2" ];then
		echo $1
	fi	
	date_str=$(date "+%e.%m.%y %T.%3N")
	echo $date_str"        "$1 >> $LOG_FILE
}



#	Созданиее ярлыка на рабочем столе 

create_shortcut(){
	PATH_TO_DESK=$1
	if [[ -d $PATH_TO_DESK	]]; then
		case $2 in 
		"--install" | "-i")
			log "Create desktop shortcut '$PATH_TO_DESK/assistant.desktop'"
			cp -p $SHORTCUT_PATH "$PATH_TO_DESK/" 2>> $LOG_FILE
			if [ $? != 0 ]; then log "Error creating shortcut"; fi
		;;
		"--uninstall" | "-u")
			rm -f "$PATH_TO_DESK/assistant.desktop"
			log "Remove desktop shortcut '$PATH_TO_DESK/assistant.desktop'"
		;;
		*)
			# echo "Invalid params: \"$2\" "
			log "Invalid params: \"$2\" " "echo"
		;;
		esac
	fi
}

add_shortcut_to_app_list(){
case $1 in
	"--install" | "-i")
		[[ -d /usr/share/applications ]] && cp -p $INST_DIR/scripts/assistant.desktop /usr/share/applications/
		if [ $? == 0 ]; then 
			# echo "Added shortcut to application list"
			log "Added shortcut to application list" "echo"
		else 
			log "Failed to add shortcut to application list"
		fi
		rm -f $SHORTCUT_PATH
		log "Old shortcut removed from application directory '$SHORTCUT_PATH'"
	;;
	"--uninstall" | "-u")
		rm -f /usr/share/applications/assistant.desktop
		log "Removed shortcut from the application list '/usr/share/applications/assistant.desktop'"
		touch $SHORTCUT_PATH
	;;
	esac
}

shortcut(){
for HOME_USER in ${USERS_LIST[@]}
	do
		DIRS=$HOME_USER$PATH_TO_CONF
		if [[ -f $DIRS ]]; then
			. $DIRS
			log "Get user data from '$DIRS'"
			DESKTOP_NAME=$(basename "$XDG_DESKTOP_DIR")
			create_shortcut "$HOME_USER$DESKTOP_NAME" $1
		fi
	done
add_shortcut_to_app_list $1
}

#	Cоздание ссылки сертификата
make_ca()
{
mkdir -p /etc/assistant/cert/
for cert in ${PATHS_TO_CERT[@]}
	do
		if [ -f $cert ] && [ ! -L $cert ]; then 
			log "'$cert' is found"
			ln -s $cert $CERT_PATH
			log "Created CA-certificate link '$CERT_PATH'" "echo"
			return 0;
		fi
	done
	log "Failed to create CA certificate link for this system" "echo"
	return 1
}

cert() {
case $1 in 
	"--install" | "-i")
		log "Adding CA certificate link"
		[ -L $CERT_PATH ] || ( make_ca && echo "CA certificate installed" )
		if [ $? != 0 ]; then log "Error adding CA certificate link"; else log "CA certificate installed"; fi
	;;
	"--uninstall" | "-u")
		log "Removing CA certificate link"
		[ ! -L $CERT_PATH ] || ( rm -r /etc/assistant && echo "CA certificate deleted" )
		if [ $? != 0 ]; then log "Error removing CA certificate link"; else log "CA certificate deleted" ; fi
	;;
	*)
		# echo "Invalid params: '$*'"
		log "Invalid params: '$*'" "echo"
	;;
esac
}

#Внесение изменений в файл "/etc/X11/trusted"
trusted(){
	case $1 in
	"--install" | "-i")
		if [[ -d $TRUST_D ]]; then
			echo "/opt/assistant/bin/assistant(ALL_R)" > $TRUST_D/assistant
			echo "/opt/assistant/bin/asts(ALL_R)" >> $TRUST_D/assistant
			echo "/opt/assistant/bin/astrct(ALL_R)" >> $TRUST_D/assistant
			log "Added entries to the file '$TRUST_D/assistant'" "echo"
		elif [[ -f $TRUST ]]; then
			sed -i '/^\/opt\/assistant\/bin\/.*/d' $TRUST
			echo "/opt/assistant/bin/assistant(ALL_R)" >> $TRUST
			echo "/opt/assistant/bin/asts(ALL_R)" >> $TRUST
			echo "/opt/assistant/bin/astrct(ALL_R)" >> $TRUST

			# echo "Added entries to the file '$TRUST'"
			log "Added entries to the file '$TRUST'" "echo"
		fi
	;;
	"--uninstall" | "-u")
		[[ -d $TRUST_D ]] && rm -f $TRUST_D/assistant
		[[ -f $TRUST ]] && sed -i '/^\/opt\/assistant\/bin\/.*/d' $TRUST
		log "Deleted entries from the file trusted file" "echo"
	;;
	esac
}

set_shell_ini(){
	case $1 in
	"--install" | "-i")
		[ -f $SHELLS ] || echo -e "[MAIN]\n" > $SHELLS
		
		if [ $ID==astra ] && [ -x /usr/bin/fly-term ]; then
			sed -i '/^fly-term=\/usr\/bin\/fly-term/d' $SHELLS
			echo "fly-term=/usr/bin/fly-term" >> $SHELLS
		fi
		chmod 777 $SHELLS
	;;
	esac
}

add_libudev_links(){
	case $1 in
	"--install" | "-i")
			if [ -z "$v_libudev" ]
			then
			    v_libudev=$(exec -c ldconfig -p | grep "libudev.so.1 " | grep "x86-64")
			    if [ -z "$v_libudev" ];then
					v_libudev=$(exec -c ldconfig -p | grep "libudev.so.0 " | grep "x86-64") 
			    fi

			    if [ -z "$v_libudev" ];then
					log "Cant find libudev.so NOR libudev.so.0 NOR libudev.so.1" "echo" 
			    else
				v_line_count=$(wc -l <<< "${v_libudev}")
				if [ $v_line_count -eq 1 ];then
					v_libudev_full=$v_libudev
				    v_libudev=${v_libudev#* => }
				    if [ -f "$v_libudev" ];then
						ln -sf "$v_libudev" "$INST_DIR/lib/libudev.so"
						if [ $? -eq 0 ];then
							    log "Symlink \"libudev\" created."
							else
							    log "Fail create symlink" "echo"
						fi
				    else
						log "Fail extract file path: \"$v_libudev_full\""
				    fi
				else
				    log "After filtering still lefts ($v_line_count) variants:" "echo"
				    log "{$v_libudev}"
				    log "Cant decide what line to select..." "echo"
				fi
			    fi
			else
			    log "libudev.so file/link already exist in system." "echo"
			fi
			;;
	"--uninstall" | "-u")
			rm -f $INST_DIR/lib/libudev.so
		;;
		esac
}

disable_pam_lightdm(){
case $1 in
	"--install" | "-i")
		changed=false
		if [ "$BUILD_ID" == "ALT SP Workstation 10" ]; then
			log "Configuring PAM for $BUILD_ID" "echo"
			for set in ${PAM_LIGHTDM_SETTINGS[@]}
			do
				if [ -f $set ]; then
					if egrep -q "^session\s+required\s+pam_namespace.so" $set; then 
						sed -i 's/^session\s\+required\s\+pam_namespace\.so/#&/' $set
						log "file '$set' adjusted"
						changed=true
					fi
				fi
			done
			if systemctl is-active lightdm &> /dev/null && $changed; then
				echo
				log "For all changes to take effect, you must re-authorize in the system." "echo"
				log "\"systemctl restart lightdm\"" "echo"
			fi
		fi
		;;
	"--uninstall" | "-u")
			:
		;;
		esac
}

if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
	# echo "Setup Assistant for '$ID $VER'"
	log "Setup Assistant for '$NAME $VERSION'" "echo"
	# log "Setup Assistant" "echo"
	log "Setup daemon"
	$INST_DIR/scripts/daemon.sh $1 | tee -a $LOG_FILE
	log "Daemon setup completed"
	add_libudev_links $1
	cert $1
	shortcut $1 
	trusted $1
	set_shell_ini $1
	disable_pam_lightdm $1
	log "Assistant setup completed" "echo"
	exit 0
fi
